/* Include System Libraries */
#include <string.h>	/* String Library */
/* Include Local Libraries */
#include "common.h"	/* Commonly Used Libraries */
#include "defs.h"	/* Global Defines */
#include "draw.h"	/* Draw Functions */
#include "message.h"	/* Message Functions */

/* Draw Help Screen */
void drawHelpScreen(WINDOW *mainWindow, unsigned int goal)
{
	/* Erase the screen */
	werase(mainWindow);
	/* Move into position (accounting for border) */
	wmove(mainWindow, 1, 1);
	/* Print program name/version/copyright/author to screen */
	wprintw(mainWindow, "Betcha! v%s\n (c) %s\n\n",
				version, copyright);
	/* Print the help instructions to screen */
	wprintw(mainWindow, " A ball-in-a-cup guessing game.\n");
	wprintw(mainWindow, " Use left/right to select cup.\n");
	wprintw(mainWindow, " Use up/down to raise/lower bet.\n");
	wprintw(mainWindow, " Press Enter to confirm choice.\n\n");
	wprintw(mainWindow, " If correct, you win double bet.\n");
	wprintw(mainWindow, " If wrong, you lose your bet.\n");
	wprintw(mainWindow, " If you reach $%u, victory!\n", goal);
	wprintw(mainWindow, " If you run out, game over!\n");
	wprintw(mainWindow, " If you are victorious, skill up!\n");
	wprintw(mainWindow, " (Higher Skill = Higher Target)\n\n");
	wprintw(mainWindow, " Press 'q' to quit.\n");
	wprintw(mainWindow, " Press 'h' for help screen.\n\n");
	wprintw(mainWindow, " Good luck!");
	/* Draw the border and refresh the screen */
	box(mainWindow, 0, 0);
	wrefresh(mainWindow);
}

void drawSkillScreen(WINDOW *mainWindow, unsigned int skill)
{
	/* Erase the screen */
	werase(mainWindow);
	/* Move into position (accounting for border) */
	wmove(mainWindow, 1, 1);
	/* Print program name/version/copyright/author to screen */
	wprintw(mainWindow, "Betcha! v%s\n (c) %s\n\n",
				version, copyright);
	/* Print skill selection header */
	wprintw(mainWindow, " SELECT SKILL LEVEL\n");
	/* Print the difficulty options */
	wattron(mainWindow, COLOR_PAIR(PAIR_BR_GREEN));
	wprintw(mainWindow, "   Easiest\n");
	wprintw(mainWindow, "   Easier\n");
	wprintw(mainWindow, "   Easy\n");
	wattroff(mainWindow, COLOR_PAIR(PAIR_BR_GREEN));
	wattron(mainWindow, COLOR_PAIR(PAIR_BR_YELLOW));
	wprintw(mainWindow, "   Medium\n");
	wattroff(mainWindow, COLOR_PAIR(PAIR_BR_YELLOW));
	wattron(mainWindow, COLOR_PAIR(PAIR_BR_RED));
	wprintw(mainWindow, "   Hard\n");
	wprintw(mainWindow, "   Harder\n");
	wprintw(mainWindow, "   Hardest\n\n");
	wattroff(mainWindow, COLOR_PAIR(PAIR_BR_RED));
	/* Print instructions to screen*/
	wprintw(mainWindow, " Use up/down to select.\n");
	wprintw(mainWindow, " Press Enter to confirm.\n");
	wprintw(mainWindow, " Press 'q' to quit.");
	/* Print cursor to screen */
	mvwprintw(mainWindow, 5+skill, 2, ">");
	/* Draw the border and refresh the screen */
	box(mainWindow, 0, 0);
	wmove(mainWindow, skill, skill);
	wrefresh(mainWindow);
}

/* Draw Main Window */
void drawMainScreen(WINDOW *mainWindow, WINDOW *cupWindow,
		WINDOW *statWindow, WINDOW *msgWindow, struct Game *game)
{
	/* Erase the screen */
	werase(mainWindow);
	/* Draw the cup window and border */
	drawRow(cupWindow, 0, 1); /* Pass zero for choice (default row) */
	drawArrow(cupWindow, game->choice);
	box(cupWindow, 0, 0);
	/* Draw player stats */
	drawPlayerStats(statWindow, game);
	/* Draw a border around the message window */
	box(msgWindow, 0, 0);
	/* Refresh the screen */
	wrefresh(mainWindow);
}

/* Draw Player Stats */
void drawPlayerStats(WINDOW *statWindow, struct Game *game)
{
	/* Erase the screen */
	werase(statWindow);
	/* Move into position (accounting for border) */
	wmove(statWindow, 1, 1);
	/* Print player's current skill */
	wprintw(statWindow, "SKILL\n ");
	/* Set text colour based on skill */
	if (game->skill < SKILL_MEDIUM)
		wattron(statWindow, COLOR_PAIR(PAIR_BR_GREEN));
	else if (game->skill > SKILL_MEDIUM)
		wattron(statWindow, COLOR_PAIR(PAIR_BR_RED));
	else
		wattron(statWindow, COLOR_PAIR(PAIR_BR_YELLOW));
	/* Output skill level */
	switch (game->skill) {
		case SKILL_EASIEST:
			wprintw(statWindow, "Easiest");
			break;
		case SKILL_EASIER:
			wprintw(statWindow, "Easier");
			break;
		case SKILL_EASY:
			wprintw(statWindow, "Easy");
			break;
		case SKILL_MEDIUM:
			wprintw(statWindow, "Medium");
			break;
		case SKILL_HARD:
			wprintw(statWindow, "Hard");
			break;
		case SKILL_HARDER:
			wprintw(statWindow, "Harder");
			break;
		case SKILL_HARDEST:
			wprintw(statWindow, "Hardest");
			break;
		default:
			wprintw(statWindow, "???");
			break;
	}
	/* Disable text colour after */
	if (game->skill < SKILL_MEDIUM)
		wattroff(statWindow, COLOR_PAIR(PAIR_BR_GREEN));
	else if (game->skill > SKILL_MEDIUM)
		wattroff(statWindow, COLOR_PAIR(PAIR_BR_RED));
	else
		wattroff(statWindow, COLOR_PAIR(PAIR_BR_YELLOW));
	wprintw(statWindow, "\n");
	/* Print player's current money */
	wprintw(statWindow, " MONEY\n");
	wprintw(statWindow, " $%u\n", game->money);
	/* Print player's current bet during gameplay */
	if (game->money > 0 && game->money < game->goal)
	{
		wprintw(statWindow, " BET\n");
		wprintw(statWindow, " $%u\n", game->bet);
	}
	/* Print border and refresh window */
	box(statWindow, 0, 0);
	wrefresh(statWindow);
}

/* Draw a message to the message window */
void drawMessage(WINDOW *msgWindow, unsigned int msgID, unsigned int pair)
{
	/* Erase the window contents */
	werase(msgWindow);
	/* Copy data */
	char msg[30];
	strcpy(msg, message[msgID]);
	/* Print to screen */
	if (pair > 0)
		wattron(msgWindow, COLOR_PAIR(pair));
	mvwprintw(msgWindow, 1, 1, "%s", msg);
	if (pair > 0)
		wattroff(msgWindow, COLOR_PAIR(pair));
	/* Draw border and refresh */
	box(msgWindow, 0, 0);
	wrefresh(msgWindow);
}

/* Draw Result Message */
void drawResultMessage(WINDOW *msgWindow, struct Game *game)
{
	/* Erase the window contents */
	werase(msgWindow);
	/* Move into position (accounting for border) */
	wmove(msgWindow, 1, 1);
	/* Create char arrays to hold data */
	char msg[30];
	char output[5] = "";
	/* If player's choice is correct... */
	if (game->choice == game->correct)
	{
		/* ...print win message and money won */
		strcpy(msg, message[MSG_WIN]);
		wprintw(msgWindow, "%s", msg);
		strcpy(msg, "");
		strcpy(msg, message[MSG_MONEY_WON]);
		sprintf(output, "%u", (game->bet*2));
		strcat(msg, output);
		strcat(msg, "!");
		wattron(msgWindow, COLOR_PAIR(PAIR_BR_GREEN));
		wprintw(msgWindow, "%s", msg);
		wattroff(msgWindow, COLOR_PAIR(PAIR_BR_GREEN));
	}
	/* If player's choice is incorrect... */
	else
	{
		/* ...print lose message and money lost */
		strcpy(msg, message[MSG_LOSE]);
		wprintw(msgWindow, "%s", msg);
		strcpy(msg, "");
		strcpy(msg, message[MSG_MONEY_LOST]);
		sprintf(output, "%u", game->bet);
		strcat(msg, output);
		strcat(msg, "!");
		wattron(msgWindow, COLOR_PAIR(PAIR_BR_RED));
		wprintw(msgWindow, "%s", msg);
		wattroff(msgWindow, COLOR_PAIR(PAIR_BR_RED));
	}
	/* Draw border and refresh window */
	box(msgWindow, 0, 0);
	wrefresh(msgWindow);
}

/* Draw Cup */
void drawCup(WINDOW *cupWindow, unsigned int y, unsigned int x)
{
	mvwprintw(cupWindow, y, x, "  _");
	mvwprintw(cupWindow, y+1, x, " / \\");
	mvwprintw(cupWindow, y+2, x, "/___\\");
}

/* Draw Row of Cups (including ball if necessary) */
void drawRow(WINDOW *cupWindow, unsigned int choice, unsigned int correct)
{
	/* Create and set y/x position variables */
	unsigned int y = 4;
	unsigned int x = 2;
	/* Erase window contents */
	werase(cupWindow);
	/* Use choice to determine position */
	switch(choice) {
		/* Draw left cup raised, other two lowered */
		case 1:
			/* Left cup */
			y = 2;
			drawCup(cupWindow, y, x);
			/* Middle cup */
			y = 4;
			x = 8;
			drawCup(cupWindow, y, x);
			/* Right cup */
			x = 14;
			drawCup(cupWindow, y, x);
			/* Set ball x position */
			x = 4;
			break;
		/* Draw middle cup raised, other two lowered */
		case 2:
			/* Left cup */
			drawCup(cupWindow, y, x);
			/* Middle cup */
			y = 2;
			x = 8;
			drawCup(cupWindow, y, x);
			/* Right cup */
			y = 4;
			x = 14;
			drawCup(cupWindow, y, x);
			/* Set ball x position */
			x = 10;
			break;
		/* Draw right cup raised, other two lowered */
		case 3:
			/* Left cup */
			drawCup(cupWindow, y, x);
			/* Middle cup */
			x = 8;
			drawCup(cupWindow, y, x);
			/* Right cup */
			y = 2;
			x = 14;
			drawCup(cupWindow, y, x);
			/* Set ball x position */
			x = 16;
			break;
		/* Draw all cups lowered */
		default:
			/* Left cup */
			drawCup(cupWindow, y, x);
			/* Middle cup */
			x = 8;
			drawCup(cupWindow, y, x);
			/* Right cup */
			x = 14;
			drawCup(cupWindow, y, x);
			break;
	}
	/* If choice is correct... */
	if (choice == correct)
	{
		/* ...set ball y position and draw the ball */
		y = CUPWIN_HEIGHT - 2;
		drawBall(cupWindow, y, x);
	}
	/* Draw border and refresh */
	box(cupWindow, 0, 0);
	wrefresh(cupWindow);
}

/* Draw Ball */
void drawBall(WINDOW *cupWindow, unsigned int y, unsigned int x)
{
	wattron(cupWindow, COLOR_PAIR(PAIR_BR_BLUE));
	mvwprintw(cupWindow, y, x, "o");
	wattroff(cupWindow, COLOR_PAIR(PAIR_BR_BLUE));
}

/* Draw Arrow */
void drawArrow(WINDOW *cupWindow, unsigned int choice)
{
	/* Set arrow position based on current choice */
	unsigned int x = 1;
	switch (choice) {
		case 1:
			/* Draw above left cup */
			x = 4;
			break;
		case 2:
			/* Draw above middle cup */
			x= 10;
			break;
		case 3:
			/* Draw above right cup */
			x = 16;
			break;
		default:
			break;
	}
	/* Draw the arrow */
	mvwprintw(cupWindow, 2, x, "|");
	mvwprintw(cupWindow, 3, x, "V");
	/* Refresh the window */
	wrefresh(cupWindow);
}

