/* Include Local Libraries */
#include "common.h"	/* Commonly Used Libraries */
#include "defs.h"	/* Global Defines */
#include "game.h"	/* Gameplay Functions */
#include "draw.h"	/* Drawing Functions */
#include "message.h"	/* Message Functions */

/* Keep Money in Bounds */
void moneyBounds(struct Game *game)
{
	if (game->money > MAX_MONEY)
		game->money = MAX_MONEY;
}

/* Keep Skill in Bounds */
void skillBounds(unsigned int *skill)
{
	if (*skill > MAX_SKILL)
		*skill = MAX_SKILL;
}

/* Set the skill level */
unsigned int setSkill(WINDOW *mainWindow, unsigned int input)
{
	/* Define variables */
	unsigned int skill = 0;
	bool confirm = false;
	/* Wait for player to confirm */
	while (!confirm)
	{
		/* Draw to screen */
		drawSkillScreen(mainWindow, skill);
		/* Get and process player input */
		input = getch();
		switch(input) {
			/* Select choice from menu */
			case KEY_UP:
				if (skill > 0)
					skill--;
				break;
			case KEY_DOWN:
				if (skill < MAX_SKILL)
					skill++;
				break;
			/* Confirm choice selected */
			case KEY_ENTER:
			case '\r':
			case '\n':
				confirm = true;
				break;
			/* Exit program */
			case 'q':
			case 'Q':
				exit(EXIT_SUCCESS);
			default:
				break;
		}
	}
	return skill;
}

/* Set the goal amount */
unsigned int setGoal(unsigned int skill)
{
	int goal = 100;
	switch (skill) {
		case SKILL_EASIEST:
			goal = 100;
			break;
		case SKILL_EASIER:
			goal = 500;
			break;
		case SKILL_EASY:
			goal = 1000;
			break;
		case SKILL_MEDIUM:
			goal = 5000;
			break;
		case SKILL_HARD:
			goal = 10000;
			break;
		case SKILL_HARDER:
			goal = 50000;
			break;
		case SKILL_HARDEST:
			goal = MAX_MONEY;
			break;
	}
	return goal;
}

/* Determine the correct answer */
unsigned int setCorrect(unsigned int correct)
{
	correct = 1 + (rand() % MAX_CHOICE);
	return correct;
}

/* New Game (Reset Game Struct) */
void newGame(struct Game *game)
{
	game->choice = 1;	/* Default choice of cup */
	game->money = 10;	/* Starting money */
	game->bet = 1;		/* Default bet */
	game->correct = setCorrect(game->correct); /* Correct answer */
	game->goal = setGoal(game->skill); /* Set goal based on skill */
}

/* End Game (Victory or Game Over) */
bool endGame(WINDOW *mainWindow, WINDOW *cupWindow, WINDOW *statWindow,
		WINDOW *msgWindow, struct Game *game, unsigned int input)
{
	/* If player runs out of money, display Game Over message */
	if (game->money <= 0)
	{
		game->money = 0;
		game->bet = 0;
		drawMessage(msgWindow, MSG_GAME_OVER, PAIR_BR_RED);
	}
	/* If player reaches the goal amount, increase player's rank and
	 * display Victory message */
	if (game->money >= game->goal)
	{
		moneyBounds(game);
		drawMessage(msgWindow, MSG_VICTORY, PAIR_BR_GREEN);
	}
	/* Draw player stats window */
	drawPlayerStats(statWindow, game);
	/* Redraw the cups */
	drawRow(cupWindow, 0, game->correct);
	/* Wait for player input */
	getch();
	/* If player beats game at max rank,
	 * output a special message */
	if (game->money >= game->goal)
	{
		/* If player beats game at the highest
		 * difficulty, output a special message */
		if (game->skill == MAX_SKILL)
		{
			drawMessage(msgWindow, MSG_MAX_WIN_1,
					PAIR_BR_GREEN);
			getch();
			drawMessage(msgWindow, MSG_MAX_WIN_2,
					PAIR_BR_GREEN);
			getch();
		}
		/* If player beats game at a lower difficulty,
		 * ask them if they want to increase it */
		else
		{
			drawMessage(msgWindow, MSG_SKILL_UP, 0);
			if(checkYesNo(input))
			{
				game->skill++;
				drawPlayerStats(statWindow, game);
			}
		}
	}
	/* Ask if player wants to play again */
	drawMessage(msgWindow, MSG_PLAY_AGAIN, 0);
	if(checkYesNo(input))
	{
		/* If player says yes, reset the game data and
		 * then redraw the main window */
		newGame(game);
		drawMainScreen(mainWindow, cupWindow,
				statWindow, msgWindow,
				game);
		return false;
	}
	else
	{
		return true;
	}
}

/* Process Player Input */
bool processInput(WINDOW *mainWindow, WINDOW *cupWindow,
			WINDOW *statWindow, WINDOW *msgWindow,
			struct Game *game, unsigned int input)
{
	/* Process the player input */
	switch (input) {
		/* Choose cup */
		case KEY_LEFT:
			if (game->choice > MIN_CHOICE)
			{
				game->choice--;
				drawRow(cupWindow, 0, game->correct);
				drawArrow(cupWindow, game->choice);
			}
			break;
		case KEY_RIGHT:
			if (game->choice < MAX_CHOICE)
			{
				game->choice++;
				drawRow(cupWindow, 0, game->correct);
				drawArrow(cupWindow, game->choice);
			}
			break;
		/* Raise/lower bet */
		case KEY_UP:
			if (game->bet < game->money)
			{
				game->bet++;
				drawPlayerStats(statWindow, game);
			}
			break;
		case KEY_DOWN:
			if (game->bet > 1)
			{
				game->bet--;
				drawPlayerStats(statWindow, game);
			}
			break;
		/* Confirm bet */
		case KEY_ENTER:
		case '\n':
		case '\r':
			/* Redraw the cup window */
			drawRow(cupWindow, game->choice, game->correct);
			/* Calculate the result and output a message */
			calcResult(game);
			drawResultMessage(msgWindow, game);
			/* Wait for player input */
			getch();
			/* Reset the choice and bet values to default
			 * and generate a new correct answer */
			game->choice = 1;
			game->bet = 1;
			game->correct = setCorrect(game->correct);
			/* Redraw the main window and subwindows */
			drawMainScreen(mainWindow, cupWindow, statWindow,
					msgWindow, game);
			break;
		/* Draw help screen */
		case 'h':
		case 'H':
			drawHelpScreen(mainWindow, game->goal);
			getch();
			/* Redraw the main window and subwindows */
			drawMainScreen(mainWindow, cupWindow, statWindow,
					msgWindow, game);
			break;
		/* Quit the game */
		case 'q':
		case 'Q':
			drawMessage(msgWindow, MSG_QUIT, 0);
			if(checkYesNo(input))
				return true;
			drawMessage(msgWindow, MSG_EMPTY, 0);
			break;
		/* Default action */
		default:
			return false;
	}
	/* Non-void function must return a value */
	return false;
}

/* Calculate Result */
void calcResult(struct Game *game)
{
	/* If the choice is correct, double player's bet as winnings */
	if (game->choice == game->correct)
		game->money += (game->bet * 2);
	/* Otherwise, player loses amount that was bet */
	else
		game->money -= game->bet;
	/* Keep player money from going beyond the maximum amount */
	moneyBounds(game);
}

/* Check Answer To Yes/No Question */
bool checkYesNo(unsigned int input)
{
	bool answer = false;
	while (!answer)
	{
		input = getch();
		switch (input) {
			case 'y':
			case 'Y':
				return true;
			case 'n':
			case 'N':
				return false;
			default:
				break;
		}
	}
	return false;
}
