/*========================================================================
 * "Betcha!", a simple terminal-based ball-in-a-cup guessing game
 * Copyright (c) Bloyburt 2024
 *------------------------------------------------------------------------
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 *========================================================================
 */
/* Include System Libraries */
#include <unistd.h>	/* Standard C Library */
#include <getopt.h>	/* Parse Command-Line Arguments */
#include <time.h>	/* Time Library (RNG) */
/* Include Local Libraries */
#include "common.h"	/* Commonly Used Libraries */
#include "defs.h"	/* Global Defines */
#include "setup.h"	/* Setup Curses */
#include "game.h"	/* Gameplay Functions */
#include "draw.h"	/* Drawing Functions */

/* Main Function */
int main(int argc, char *argv[])
{
	/* Check for options first */
	int opt = 0;
	if (argc > 1)
	{
		/* Parse through options */
		while((opt = getopt(argc, argv, "hv")) != -1)
		{
			switch(opt) {
				case 'h':
					printf("betcha <options>\n");
					printf("  -h	This help message\n");
					printf("  -v	Print version and exit\n");
					exit(EXIT_SUCCESS);
					break;
				case 'v':
					/* Output program name and
					 * program version */
					printf("Betcha! v%s\n", version);
					exit(EXIT_SUCCESS);
					break;
				default:
					/* Exit with error report if
					 * option is invalid */
					fprintf(stderr, "Usage: %s [-vh]\n", argv[0]);
					exit(EXIT_FAILURE);
					break;
			}
		}
	}

	/* Initialise random number generator */
	time_t t;
	srand((unsigned) time(&t));

	/* Initialise curses */
	initCurses();
	
	/* Create the main window and subwindows... */
	WINDOW * mainWindow;	// Main window
	WINDOW * cupWindow;	// Cup and ball window
	WINDOW * statWindow;	// Player stats window (money and bet)
	WINDOW * msgWindow;	// Message window
	/* ...and then initialise them */
	mainWindow = newwin(WINDOW_HEIGHT, WINDOW_WIDTH, 0, 0);
	cupWindow = subwin(mainWindow, CUPWIN_HEIGHT, CUPWIN_WIDTH, 0, 0);
	statWindow = subwin(mainWindow, CUPWIN_HEIGHT, STATWIN_WIDTH,
			0, CUPWIN_WIDTH + 1);
	msgWindow = subwin (mainWindow, MSGWIN_HEIGHT,
			CUPWIN_WIDTH + STATWIN_WIDTH + 1,
			CUPWIN_HEIGHT, 0);
	
	/* Initialise quit and input variables */
	bool quit = false;
	unsigned int input = 0;

	/* Create game structure */
	struct Game game;

	/* Determine difficulty via menu option */
	game.skill = setSkill(mainWindow, input);

	/* Initialise game structure */
	newGame(&game);

	/* Display the help screen as the initial screen */
	drawHelpScreen(mainWindow, game.goal);
	getch();

	/* Draw main window and various subwindows for the first time */
	drawMainScreen(mainWindow, cupWindow, statWindow, msgWindow,
			&game);
	
	/* Main program loop */
	while (!quit)
	{
		/*/ Get player input */
		input = getch();
		quit = processInput(mainWindow, cupWindow, statWindow,
					msgWindow, &game, input);

		/* Check for Victory and Game Over conditions */
		if (game.money <= 0 || game.money >= game.goal)
			quit = endGame(mainWindow, cupWindow, statWindow,
					msgWindow, &game, input);
	}
	
	/* Exit Program */
	deleteWindows(mainWindow, cupWindow, statWindow, msgWindow);
	exitCurses();
	return(EXIT_SUCCESS);
}

